"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.runCommand = void 0;
const colors_1 = __importDefault(require("../colors"));
const op_1 = require("../op");
const log_1 = require("../util/log");
const cli_1 = require("../util/cli");
const yaml_config_1 = require("../yaml-config");
const index_1 = require("../operations/index");
const project_1 = require("@trapezedev/project");
const kleur_1 = __importDefault(require("kleur"));
async function runCommand(ctx, configFile) {
    let processed;
    let handlers = await (0, index_1.loadHandlers)();
    try {
        const config = await (0, yaml_config_1.loadYamlConfig)(ctx, configFile);
        processed = (0, op_1.processOperations)(config);
    }
    catch (e) {
        log_1.logger.error(`Unable to load config file: ${e.message}`);
        throw e;
    }
    if (!processed.length) {
        (0, log_1.warn)('No operations to apply, exiting...');
        process.exit(0);
    }
    try {
        await executeOperations(ctx, handlers, processed);
    }
    catch (e) {
        throw e;
    }
}
exports.runCommand = runCommand;
async function executeOperations(ctx, handlers, operations) {
    for (const op of operations) {
        if (!ctx.args.quiet) {
            printOp(ctx, op);
        }
        const skipped = op.platform !== 'project' && (op.platform === 'ios' ? !ctx.project.ios : !ctx.project.android);
        if (skipped) {
            project_1.Logger.debug(`Skipping ${op.id} because ${op.platform} project does not exist`);
            continue;
        }
        if (!(0, index_1.isOpRegistered)(handlers, op.id)) {
            log_1.logger.warn(`Unsupported configuration option ${colors_1.default.strong(op.id)}. Skipping`);
            continue;
        }
        (await (0, index_1.runOperation)(ctx, handlers, op)) || [];
    }
    await checkModifiedFiles(ctx);
}
function printOp(ctx, op) {
    const skipped = op.platform === 'ios' ? !ctx.project.ios : !ctx.project.android;
    // const env = c.weak(`[${op.env}]`);
    const tag = skipped ? colors_1.default.weak(colors_1.default.strong(`skip`)) : kleur_1.default.bold().magenta(`run`);
    const platform = colors_1.default.success(colors_1.default.strong(`${op.platform}`));
    const opName = colors_1.default.strong(op.name);
    const opDisplay = op.displayText;
    (0, log_1.log)(tag, platform, opName, opDisplay);
}
async function printDiff(diff) {
    var _a, _b;
    const lines = (_b = (_a = diff.patch) === null || _a === void 0 ? void 0 : _a.split(/\r?\n|\r/g)) !== null && _b !== void 0 ? _b : [];
    console.log(lines.map(line => {
        switch (line[0]) {
            case "+": return colors_1.default.success(line.trimEnd());
            case "-": return colors_1.default.log.ERROR(line.trimEnd());
            default: return line.trimEnd();
        }
    }).join('\n'));
}
async function checkModifiedFiles(ctx) {
    const files = ctx.project.vfs.all();
    const diffs = ctx.args.diff ? await ctx.project.vfs.diffAll() : [];
    Object.keys(files).map(k => {
        const file = files[k];
        (0, log_1.log)(colors_1.default.log.WARN(colors_1.default.strong(`updated`)), file.getFilename());
        const diff = diffs.find((d) => d.file === file);
        if (diff && ctx.args.diff) {
            printDiff(diff);
        }
    });
    if (ctx.args.noCommit) {
        return;
    }
    if (!ctx.args.dryRun && !ctx.args.y) {
        const answers = await (0, cli_1.logPrompt)(colors_1.default.strong(`Apply changes?\n`) +
            `Applying these changes will modify your source files. We recommend committing any changes before running this operation.`, {
            type: 'confirm',
            name: 'apply',
            message: `Apply?`,
            initial: false,
        });
        if (answers.apply) {
            return ctx.project.vfs.commitAll(ctx.project);
        }
        else {
            (0, log_1.log)('Not applying changes. Exiting');
        }
    }
    else if (!ctx.args.dryRun && ctx.args.y) {
        log_1.logger.info('-y provided, automatically applying configuration');
        return ctx.project.vfs.commitAll(ctx.project);
    }
}
//# sourceMappingURL=run.js.map